import sys
import os
from pathlib import Path
from queue import Queue
import time
import threading
from collections import namedtuple
import logging

try:
    import memryx
except ImportError:
    mix_home = os.getenv("MIX_HOME")
    if not mix_home:
        print("Install MemryX SDK or clone MIX and source setup_env.sh")
        exit(1)
    sys.path.append(mix_home)

import cv2 as cv
import numpy as np

from memryx import AsyncAccl
from face_detection.app import App as FaceApp
from emotion_recognition.app import App as EmotionApp

class FaceCropper(FaceApp):
    def __init__(self, *args, **kwargs):
        super().__init__(*args, **kwargs)
        self.model.output_map = {
            'regressor_8': 2,
            'regressor_16': 3,
            'classificator_8': 0,
            'classificator_16': 1,
        }
        self.scale = kwargs.get('scale', 1)

    def generate_frame_face(self):
        return self.generate_frame()

    def process_face(self, *ofmaps):
        dets = self.model.postprocess(*ofmaps)
        frame = self.capture_queue.get()
        if dets.size == 0:
            return frame, 0
        face = self._crop_faces(frame, dets)
        return face, dets.shape[0]

    def _crop_faces(self, frame, dets):
        # only pick the first face
        xmin, ymin, xmax, ymax = self._get_box(dets[0])
        xmin = np.clip(xmin, 0, frame.shape[1])
        ymin = np.clip(ymin, 0, frame.shape[0])
        xmax = np.clip(xmax, 0, frame.shape[1])
        xmin = np.clip(xmin, 0, frame.shape[0])
        if self.scale != 1:
            face_height = ymax - ymin
            face_width = xmax - xmin
            p = 0.5*(self.scale - 1)
            ymin = np.clip(int(ymin - p*face_height), 0, frame.shape[0])
            ymax = np.clip(int(ymax + p*face_height), 0, frame.shape[0])
            xmin = np.clip(int(xmin - p*face_width), 0, frame.shape[1])
            xmax = np.clip(int(xmax + p*face_width), 0, frame.shape[1])
        return frame[ymin : ymax + 1, xmin : xmax + 1, :]


class EmotionWithBackground(EmotionApp):
    def __init__(self, *args, **kwargs):
        super().__init__(*args, **kwargs)
        self.idx_to_class[7] = 'Background'
        self.idx_to_emoji[7] = self._load_emoji(self.emoji_dir/'1F50D_color.png')
        self.background = False

    def process_model_output(self, *ofmaps):
        if self.background:
            self.background = False
            emotion_idx = 7
            emotion_idx = self._smooth_emotion(emotion_idx)
        else:
            emotion_idx = self.postprocess(ofmaps[0], (self.input_width, self.input_height))
        logging.debug(self.idx_to_class[emotion_idx])
        img = self.cap_queue.get()
        out = self.draw(img, emotion_idx)
        return out

class App:
    def __init__(self, cam, smooth_duration=7, mirror=False, scale=1.2, **kwargs):
        Shape = namedtuple('Shape', ['height', 'width'])
        self.face_app = FaceCropper(cam, model_input_shape=Shape(height=128, width=128), mirror=mirror, scale=scale)
        self.emotion_app = EmotionWithBackground(cam, smooth_duration=smooth_duration, mirror=mirror)
        #self.face_done = threading.Event()
        self.face = None
        self.capture_queue = Queue()

    def generate_frame_face(self):
        frame = self.face_app.generate_frame()
        if frame is None:
            return None
        orig_frame = self.face_app.capture_queue.get()
        self.capture_queue.put(orig_frame)
        self.face_app.capture_queue.put(orig_frame)
        return frame

    def process_face(self, *ofmaps):
        if(len(ofmaps)==1):
            ofmaps = ofmaps[0]
        self.face, face_count = self.face_app.process_face(*ofmaps)
        if face_count == 0:
            self.emotion_app.background = True
        #self.face_done.set()
        return self.face

    def generate_frame_emotion(self):
        # self.face_done.wait()
        # self.face_done.clear()
        try:
            face = cv.resize(self.face, (224, 224), interpolation=cv.INTER_CUBIC)
        except Exception:
            self.emotion_app.background = True
            face = np.zeros((224,224,3))
        self.emotion_app.cap_queue.put(self.capture_queue.get())
        return face.astype(np.float32)

    def process_emotion(self, *ofmaps):
        out = self.emotion_app.process_model_output(*ofmaps)
        self.emotion_app.show(out)
        return out

def run_mxa(dfp):
    accl = AsyncAccl(dfp)
    accl.connect_input(app.generate_frame_face)
    accl.connect_input(app.generate_frame_emotion, 1)
    accl.connect_output(app.process_face)
    accl.connect_output(app.process_emotion, 1)
    accl.wait()

if __name__ == '__main__':
    cam = cv.VideoCapture('/dev/video0')
    parent_path = Path(__file__).resolve().parent 
    app = App(cam, mirror=True)
    dfp = parent_path / 'models.dfp'
    run_mxa(dfp)
