import sys
import os
from pathlib import Path
from queue import Queue
import time
from collections import Counter, deque
import logging

try:
    import memryx
except ImportError:
    mix_home = os.getenv("MIX_HOME")
    if not mix_home:
        print("Install MemryX SDK or clone MIX and source setup_env.sh")
        exit(1)
    sys.path.append(mix_home)

import cv2 as cv
import numpy as np

from memryx import AsyncAccl

class App:
    def __init__(self, cam, smooth_duration=7, mirror=False, **kwargs):
        self.cam = cam
        self.input_height = int(cam.get(cv.CAP_PROP_FRAME_HEIGHT))
        self.input_width = int(cam.get(cv.CAP_PROP_FRAME_WIDTH))
        self.mirror = mirror
        self.cap_queue = Queue()
        self.idx_to_class = {
            0: 'Anger',
            1: 'Disgust',
            2: 'Fear',
            3: 'Happiness',
            4: 'Neutral',
            5: 'Sadness',
            6: 'Surprise'
        }
        self.emoji_dir = Path(__file__).resolve().parent / 'emojis'
        self.idx_to_emoji = {
            0: self._load_emoji(self.emoji_dir/'1F92C_color.png'),
            1: self._load_emoji(self.emoji_dir/'1F92E_color.png'),
            2: self._load_emoji(self.emoji_dir/'1F628_color.png'),
            3: self._load_emoji(self.emoji_dir/'1F604_color.png'),
            4: self._load_emoji(self.emoji_dir/'1F610_color.png'),
            5: self._load_emoji(self.emoji_dir/'1F622_color.png'),
            6: self._load_emoji(self.emoji_dir/'1F62F_color.png'),
        }
        self.horz_margin = 150
        self.vert_margin = 20

        self.emotion_queue = deque()
        self.emotion_ctr = Counter()
        self.emotion_duration = smooth_duration

    def _load_emoji(self, path, shape=(128, 128)):
        emoji = cv.imread(str(path), cv.IMREAD_UNCHANGED)
        return cv.resize(emoji, shape, cv.INTER_CUBIC)

    def generate_frame(self):
        ok, frame = self.cam.read()
        if not ok:
            return None
        if self.mirror:
            frame = cv.flip(frame, 1)
        self.cap_queue.put(frame)
        return self.preprocess(frame)

    def process_model_output(self, *ofmaps):
        emotion_idx = self.postprocess(ofmaps[0], (self.input_width, self.input_height)) 
        logging.debug(self.idx_to_class[emotion_idx])
        img = self.cap_queue.get()
        out = self.draw(img, emotion_idx)
        self.show(out)
        return out

    def _smooth_emotion(self, emotion_idx):
        if len(self.emotion_queue) == self.emotion_duration:
            oldest = self.emotion_queue.popleft()
            self.emotion_ctr[oldest] -= 1
            if self.emotion_ctr[oldest] == 0:
                del self.emotion_ctr[oldest]
        self.emotion_queue.append(emotion_idx)
        self.emotion_ctr[emotion_idx] += 1
        argmax = None
        max_count = 0
        for idx, count in self.emotion_ctr.items():
            if count > max_count:
                max_count = count
                argmax = idx
        return argmax

    def preprocess(self, img):
        if self.mirror:
            img = cv.flip(img, 1)
        center = img[self.vert_margin : -self.vert_margin, self.horz_margin : -self.horz_margin, :]
        arr = np.array(cv.resize(center, (224, 224))).astype(np.float32)
        arr[..., 0] -= 103.939
        arr[..., 1] -= 116.779
        arr[..., 2] -= 123.68
        return arr

    def draw(self, img, idx):
        emoji = self.idx_to_emoji[idx]
        h, w, c = emoji.shape
        ratio = 0.75
        alpha = 1 - np.expand_dims(emoji[:, :, -1]/255.0, -1)
        alpha = np.repeat(alpha, 3, axis=-1).astype(np.float32)
        fg = cv.multiply(alpha, img[:h, :w, :].astype(np.float32))
        bg = cv.multiply((1 - alpha), emoji[:, :, :-1].astype(np.float32))
        img[:h, :w, :] = cv.add(fg, bg)
        return img

    def show(self, img):
        cv.imshow('Emotions', img)
        if cv.waitKey(1) == ord('q'):
            cv.destroyAllWindows()
            exit(1)

    def postprocess(self, logits, original_shape):
        emotion_idx = np.argmax(np.squeeze(logits))
        emotion_idx = self._smooth_emotion(emotion_idx)
        return emotion_idx

def run_mxa(dfp):
    accl = AsyncAccl(dfp)
    accl.connect_input(app.generate_frame)
    accl.connect_output(app.process_model_output)
    accl.wait()

if __name__ == '__main__':
    cam = cv.VideoCapture('/dev/video0')
    parent_path = Path(__file__).resolve().parent 
    app = App(cam, mirror=True)
    dfp = parent_path / 'mobilenet_7.dfp'
    run_mxa(dfp)
