import sys
import os
import subprocess
import shutil
from pathlib import Path
from abc import ABC, abstractmethod
import time
import threading
from queue import Queue
from collections import namedtuple

import numpy as np
import cv2 as cv

sys.path.append(str(Path(__file__).resolve().parent))
import model

try:
    import memryx
except ImportError:
    mix_home = os.getenv("MIX_HOME")
    if not mix_home:
        print("Install MemryX SDK or clone MIX and source setup_env.sh")
        exit(1)
    sys.path.append(mix_home)

from memryx import AsyncAccl

class App:
    def __init__(self, cam, model_input_shape, mirror=False, **kwargs):
        self.cam = cam
        self.input_height = int(cam.get(cv.CAP_PROP_FRAME_HEIGHT))
        self.input_width = int(cam.get(cv.CAP_PROP_FRAME_WIDTH))
        self.model_input_shape = model_input_shape
        self.capture_queue = Queue()
        self.mirror = mirror
        self.model = model.MPFaceDetector(model_input_shape)

    def generate_frame(self):
        ok, frame = self.cam.read()
        if not ok:
            print('EOF')
            return None
        if self.mirror:
            frame = cv.flip(frame, 1)
        self.capture_queue.put(frame)
        out = self.model.preprocess(frame)
        return out

    def process_model_output(self, *ofmaps):
        dets = self.model.postprocess(*ofmaps)
        frame = self.capture_queue.get()
        out = self.draw(frame, dets)
        self.show(out)

    def draw(self, img, dets):
        for i in range(dets.shape[0]):
            xmin, ymin, xmax, ymax = self._get_box(dets[i])
            img = cv.rectangle(img, (xmin, ymin), (xmax, ymax), (255,0,0), 3)  
        return img

    def _get_box(self, det):
        ymin = int(det[0]*self.input_height)
        xmin = int(det[1]*self.input_width)
        ymax = int(det[2]*self.input_height)
        xmax = int(det[3]*self.input_width)
        return [xmin, ymin, xmax, ymax]

    def show(self, img):
        cv.imshow('Faces', img)
        if cv.waitKey(1) == ord('q'):
            cv.destroyAllWindows()
            exit(1)
        
def run_mxa(dfp):
    accl = AsyncAccl(dfp)
    accl.connect_input(app.generate_frame)
    accl.connect_output(app.process_model_output)
    accl.wait()

if __name__ == '__main__':
    cam = cv.VideoCapture('/dev/video0')
    parent_path = Path(__file__).resolve().parent 
    dfp = parent_path / 'face_detection_short_range.dfp'
    Shape = namedtuple('Shape', ['height', 'width'])
    app = App(cam, Shape(height=128, width=128), mirror=True)
    run_mxa(dfp)
