#include <iostream>
#include <signal.h>
#include <opencv2/opencv.hpp>    /* imshow */
#include <opencv2/imgproc.hpp>   /* cvtcolor */
#include <opencv2/imgcodecs.hpp> /* imwrite */
#include "memx/accl/MxAccl.h"

namespace fs = std::filesystem;

#define AVG_FPS_CALC_FRAME_COUNT  50

const fs::path modelPath = "midas_v2_1_small.dfp";
fs::path videoPath = "../pexels_videos_2103099.mp4"; 

atomic_bool runflag(true);
void signalHandler(int pSignal){
    runflag.store(false);
}

// In case of cameras try to use best possible input configurations which are setting the
// resolution to 640x480 and try to set the input FPS to 30
bool configureCamera(cv::VideoCapture& vcap) {
    bool settings_success = true;

    try {
        if (!vcap.set(cv::CAP_PROP_FRAME_HEIGHT, 480) || 
            !vcap.set(cv::CAP_PROP_FRAME_WIDTH, 640) || 
            !vcap.set(cv::CAP_PROP_FPS, 30)) {
            std::cout << "Setting vcap Failed\n";
            cv::Mat simpleframe;
            if (!vcap.read(simpleframe)) {
                settings_success = false;
            }
        }
    } catch (...) {
        std::cout << "Exception occurred while setting properties\n";
        settings_success = false;
    }

    return settings_success;
}

// Tries to open the camera with custom settings set in configureCamera
// If not possible, open it with default settings
bool openCamera(cv::VideoCapture& vcap, int device, int api) {
    vcap.open(device, api);
    if (!vcap.isOpened()) {
        std::cerr << "Failed to open vcap\n";
        return false;
    }

    if (!configureCamera(vcap)) {
        vcap.release();
        vcap.open(device, api);
        if (vcap.isOpened()) {
            std::cout << "Reopened vcap with original resolution\n";
        } else {
            std::cerr << "Failed to reopen vcap\n";
            return false;
        }
    }

    return true;
}

class DepthEstimation{
    private:
        MX::Types::MxModelInfo model_info;
        cv::VideoCapture vcap;
        int origHeight;
        int origWidth;
        int model_input_height;
        int model_input_width;
        int model_output_height;
        int model_output_width;

        //FPS calculation variables
        int frame_count;
        float fps_number;
        string fps_text;
        chrono::milliseconds start_ms;
        
        std::string window_name;
        bool window_created;
        cv::Size displaySize;

        //incallback images
        cv::Mat img_resized;
        cv::Mat img_model_in;

        //outcallback images
        cv::Mat img_model_out_uint;
        cv::Mat img_final_output;
        cv::Mat img_final_out_resized;
        cv::Mat img_model_out;
    public:
        DepthEstimation(MX::Runtime::MxAccl* accl, bool use_cam)
        {
            model_info = accl->get_model_info(0);
            model_input_height = model_info.in_featuremap_shapes[0][0];
            model_input_width = model_info.in_featuremap_shapes[0][1];
            model_output_height = model_info.out_featuremap_shapes[0][0];
            model_output_width = model_info.out_featuremap_shapes[0][1];
            if(use_cam){
                #ifdef __linux__
                    std::cout << "Running on Linux" << "\n";
                    if (!openCamera(vcap, 0, cv::CAP_V4L2)) {
                        throw(std::runtime_error("Failed to open: camera 0"));
                    }

                #elif defined(_WIN32)
                    std::cout << "Running on Windows" << "\n";
                    if (!openCamera(vcap, 0, cv::CAP_ANY)) {
                        throw(std::runtime_error("Failed to open: camera 0"));
                    }
                #endif
            }
            else{
                vcap.open(videoPath.c_str(),cv::CAP_ANY);
            }


            if(vcap.isOpened()){
                std::cout << "videocapture opened \n";

                origWidth = vcap.get(cv::CAP_PROP_FRAME_WIDTH); //get the width of frames of the video 
                origHeight = vcap.get(cv::CAP_PROP_FRAME_HEIGHT);
                runflag.store(true);
            }
            else{
                std::cout << "videocapture NOT opened \n";
                runflag.store(false);
            }

            frame_count = 0;
            fps_number = 0;
            fps_text = "FPS = ";
            start_ms = 0ms;

            //Connecting the stream to the accl object. As the callback functions are defined as part of the class
            //DepthEstimation we should bind them with the possible input parameters
            auto in_cb = std::bind(&DepthEstimation::incallback_getframe, this, std::placeholders::_1, std::placeholders::_2);
            auto out_cb = std::bind(&DepthEstimation::outcallback_getmxaoutput, this, std::placeholders::_1, std::placeholders::_2);
            accl->connect_stream(in_cb, out_cb, 0/**unique stream idx*/, 0/**model idx*/);
            window_name = "Depth Estimation";
            window_created = false;

            img_resized.create(model_input_height, model_input_width, CV_8UC3);
            img_model_in.create(model_input_height, model_input_width, CV_32FC3);

            int display_width = (origWidth<1920) ? origWidth : 1920;
            int display_height = (origHeight<1080) ? origHeight : 1080;
            displaySize = cv::Size(display_width,display_height);
            img_model_out.create(cv::Size(model_output_height, model_output_width), CV_32FC1);
            img_model_out_uint.create(img_model_out.size(),CV_8UC1);
            img_final_output.create(img_model_out.size(),CV_8UC1);
            img_final_out_resized.create(displaySize,CV_8UC3);

        }
        // Input callback function
        bool incallback_getframe(vector<const MX::Types::FeatureMap<float>*> dst, int streamLabel){

            if(runflag.load()){
                cv::Mat inframe;
                bool got_frame = vcap.read(inframe);

                if (!got_frame) {
                    std::cout << "No frame \n\n\n";
                    runflag.store(false);
                    return false;  // return false if frame retrieval fails
                }
                else{
                
                    // resize to model size
                    cv::resize(inframe, img_resized, img_resized.size());

                    // convert to RGB
                    cv::cvtColor(img_resized, img_resized, cv::COLOR_BGR2RGB);

                    // convert to FP32 [0,1]
                    
                    img_resized.convertTo(img_model_in, CV_32FC3, 1.0 / 255.0);

                    // apply the MiDAS normalization constants
                    cv::add(img_model_in, cv::Scalar(-0.485, -0.456, -0.406), img_model_in);
                    cv::multiply(img_model_in, cv::Scalar(1.0/0.229, 1.0/0.224, 1.0/0.225), img_model_in);

                    // Set preprocessed input data to be sent to accelarator
                    dst[0]->set_data((float*)img_model_in.data, false);

                    return true;
                }           
            }
            else{
                vcap.release();
                return false;
            }    
        }

        // Output callback function
        bool outcallback_getmxaoutput(vector<const MX::Types::FeatureMap<float>*> src, int streamLabel){
            //get output data from accelarator
            src[0]->get_data((float *)img_model_out.data, false);

            double depth_min_d, depth_max_d;
            float depth_min, depth_max;
            cv::minMaxIdx(img_model_out, &depth_min_d, &depth_max_d);
            depth_min = (float)depth_min_d;
            depth_max = (float)depth_max_d;
            float diff = depth_max - depth_min;

            // do the scaling
            cv::add(img_model_out, cv::Scalar(-1.0 * depth_min), img_model_out);
            cv::multiply(img_model_out, cv::Scalar(1.0 / diff), img_model_out);
            cv::multiply(img_model_out, cv::Scalar(255.0), img_model_out);

            // convert to UINT8
            img_model_out.convertTo(img_model_out_uint, CV_8UC1);

            // apply colormap
            cv::applyColorMap(img_model_out_uint, img_final_output, cv::COLORMAP_INFERNO);
            //Calulate FPS once every AVG_FPS_CALC_FRAME_COUNT frames     
            frame_count++;
            if (frame_count == 1)
            {
                start_ms = std::chrono::duration_cast<std::chrono::milliseconds>(std::chrono::system_clock::now().time_since_epoch());
            }
            else if (frame_count % AVG_FPS_CALC_FRAME_COUNT == 0)
            {
                std::chrono::milliseconds duration =
                    std::chrono::duration_cast<std::chrono::milliseconds>(std::chrono::system_clock::now().time_since_epoch()) - start_ms;
                fps_number = (float)AVG_FPS_CALC_FRAME_COUNT * 1000 / (float)(duration.count());
                fps_text = "FPS ="+to_string(fps_number);
                frame_count = 0;
            }
            cv::resize(img_final_output, img_final_out_resized,displaySize);
            //Write FPS values on the display image
            cv::putText(img_final_out_resized,fps_text,
                cv::Point2i(10, 30), // origin of text (bottom left of textbox)
                cv::FONT_ITALIC,
                0.8, // font scale
                cv::Scalar(255, 255, 0), // color (green)
                2 // thickness
            );

            //Create a postion the display window in the first iteration
            if(!window_created){
                cv::namedWindow(window_name, cv::WINDOW_NORMAL | cv::WINDOW_KEEPRATIO);
                cv::resizeWindow(window_name, displaySize);
                int posx = streamLabel%4;
                int posy = streamLabel/4;
                cv::moveWindow(window_name, 50+640*posx, posy*500);
                window_created=true;
            }

            // Display the image with detections
            cv::imshow(window_name, img_final_out_resized);
            if (cv::waitKey(1) == 'q') {
                runflag.store(false);
            }     
            return true;
        }
        ~DepthEstimation(){
        }
};

int main(int argc, char* argv[]){
    bool use_cam = false;
    if(argc>1){
        std::string inputType(argv[1]);
        if(inputType == "--cam"){
            use_cam = true;
            runflag.store(true);
        }
        else if(inputType == "--video"){
            use_cam = false;
            runflag.store(true);
        }
        else{
            std::cout << "\n\nIncorrect Argument Passed \n\tuse ./depthEstimation [--cam or --video ]\n\n\n";
            runflag.store(false);
        }
    }
    else{
        std::cout << "\n\nNo Arguments Passed \n\tuse ./depthEstimation [--cam or --video ]\n\n\n";
        runflag.store(false);
    }

    signal(SIGINT, signalHandler);

    if(runflag.load()){

        std::cout << "application start \n";
        std::cout << "model path = " << modelPath.c_str() << "\n";
        if(!use_cam) { 
            std::cout << "Video File is used as input \nVideo Path = " << videoPath.c_str() << "\n"; 
        }
        
        MX::Runtime::MxAccl accl;
        int tag = accl.connect_dfp(modelPath);
        DepthEstimation app(&accl,use_cam);
        accl.start();
        accl.wait();
        accl.stop();   
        // delete accl;
    }

    else{
        std::cout << "App exiting without execution \n\n\n";       
    }

    return 1;
}
